/* ############### generated on Thu Apr 29 09:15:01 EDT 2010 ################ */

/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "ps2_protocol_keyboard.h"
#include <assert.h>

/******************************************************************************/
/*																			  */
/* 		PS/2 host keyboard and device keyboard low level PS/2 protocol routines	  */
/*																			  */
/*		A host is typically as personal computer with PS/2 connector, 		  */
/* 		and device is typically a PS/2 keyboard or a PS/2 three-button mouse  */			
/*																			  */			
/******************************************************************************/

/* -------------------------------------------------------------------------- */

enum {
	LOOP_50uS  = 10, 	/* magic constant for timing loop */
};

/* -------------------------------------------------------------------------- */

/* ========================================================================== */
/*																			  */	
/*							talk to host routines							  */
/*																			  */	
/* ========================================================================== */

/*
 * This function checks if the host keyboard port has sent a request-to-send (RTS)
 */

int is_host_keyboard_requesting_to_send(void)
{
	int timing_loop = 0;

	if (input(keyboard_host_clock) != LOW) {

		return FALSE;
	}

	output_high(MARK_PIN);
	output_high(keyboard_host_led);

	/* time if the host clock is LOW for at least 50 us */
	timing_loop = 0;		
	while (input(keyboard_host_clock) == LOW) { 
		if (++timing_loop == LOOP_50uS)
			break; 
	}
	
	output_low(MARK_PIN);
	output_low(keyboard_host_led);

	if (timing_loop < LOOP_50uS) {
		return FALSE;		/* careful: some hosts don't follow the rules */
	}

	/* continue waiting for host to release clock to HIGH */
	while (input(keyboard_host_clock) != HIGH);

#if 0 	/* the spec calls for this but this is a redundant check */
	/* check if host pulled data LOW */
	if (input(keyboard_host_data) != LOW) {
		assert(("bad data pin value", FALSE));		
		return FALSE;		
	}
#endif

	return TRUE;
}

/* -------------------------------------------------------------------------- */

/*
 * This routine receives a command code from the host keyboard port
 *
 *		-- this routine acts as a device
 *
 *		precondition: host has done a request-to-send
 */

unsigned int receive_from_host_keyboard(void)
{
	int clock;
	int start;
	int i;
	int data[10]; /* 8 data bits, one parity bit, one stop bit */
	int calc_parity = HIGH;
	int parity;
	int stop;
	unsigned int commandcode = 0;

	clock = input(keyboard_host_clock);
	assert (("bad clock value", clock == HIGH));
	start = input(keyboard_host_data);
	assert (("bad data start", start == LOW));

	/* delay a while before reading data bits */
	delay_us(100);

	output_high(MARK_PIN);
	output_high(keyboard_host_led);

	/* read the eight data bits, one parity bit, and one stop bit */
	for (i = 0; i < 10; i++) {
		output_low(keyboard_host_clock);		/* generate clock pulse */
		delay_us(37);
 
		if (input(keyboard_host_data) == HIGH) {		/* compiler bug: input() */
			data[i] = HIGH;	  /* does not like to assign to an array element */
		} else {
			data[i] = LOW;
		}

		output_high(keyboard_host_clock);		/* generate clock pulse */
		delay_us(37);
	}

	/* send the acknowledgement */
	output_low(keyboard_host_data);
	delay_us(10);
	output_low(keyboard_host_clock);
	delay_us(37);
	output_high(keyboard_host_clock);
	delay_us(20);
	output_high(keyboard_host_data);

	output_low(MARK_PIN);
	output_low(keyboard_host_led);

	parity = data[8];
	stop = data[9];

	/* calculate parity */
	for (i = 0; i < 8; i++) {
		calc_parity ^= data[i];
	}
	assert (("bad data parity", parity == calc_parity));
	assert (("bad data stop", stop == HIGH));

	for (i = 0; i < 8; i++) {
		if (data[i] == HIGH) {
			bit_set(commandcode, i);
		} else {
			bit_clear(commandcode, i);
		}
	}

	input(keyboard_host_clock);	/* set to high impedance input */

	return commandcode;
}

/* -------------------------------------------------------------------------- */

/*
 * This function waits for a request-to-send from the host keyboard port before
 * receiveing a command code from the host keyboard port
 */

unsigned int receive_from_host_keyboard_wait(void)
{

	while (is_host_keyboard_requesting_to_send() != TRUE);

	return receive_from_host_keyboard();
}

/* -------------------------------------------------------------------------- */

/*
 * This routine sends a packet to the host keyboard port 
 *
 *		-- this routine acts as a device
 */

void send_to_host_keyboard(unsigned int packet)
{
	int timing_loop = 0;
	int i;
	int ndata = 0;
	int parity = HIGH;
	int data[11]; /* one start bit, 8 data bits, one parity bit, one stop bit */
	int a_bit;

#if DEBUG == 1
	printf("packet: %02X\r\n", packet);
#endif

	/* check if communications if inhibited */
	while (input(keyboard_host_clock) == LOW);
	
	/* set start bit */
	data[ndata++] = LOW;

	/* set data bits and compute parity bit */
	for (i = 0; i < 8; i++) {
		a_bit = bit_test(packet, i);
		parity ^= a_bit;
		data[ndata++] = a_bit;
	}

	/* set partity bit */
	data[ndata++] = parity;

	/* set stop bit */
	data[ndata++] = HIGH;

	/* time if host clock is HIGH for at least 50 us before sending data */
	timing_loop = 0;		
	while (input(keyboard_host_clock) == HIGH) {   
		if (++timing_loop == LOOP_50uS)
			break; 
	}	
	/* note: assertions are for programming errors -- not bad input */
	assert (("bad clock timing", timing_loop == LOOP_50uS));

	output_high(MARK_PIN);
	output_high(keyboard_host_led);

	/* send all the bits in little endian order */
	for (i = 0; i < ndata; i++) {
		output_bit(keyboard_host_data, data[i]);
		delay_us(19);

		output_low(keyboard_host_clock);		/* generate clock pulse */
		delay_us(39);
		output_high(keyboard_host_clock);		/* generate clock pulse */
		delay_us(19);
	}

	output_low(MARK_PIN);
	output_low(keyboard_host_led);

	input(keyboard_host_clock);	/* set to high impedance input */
	input(keyboard_host_data);		/* set to high impedance input */
}

/* ========================================================================== */
/*																			  */	
/*							talk to device routines							  */
/*																			  */	
/* ========================================================================== */

/*
 * This routine sends a packet to the device
 *
 *		-- this routine acts as a host
 */

void send_to_device_keyboard(unsigned int packet)
{
	int i;
	int parity = HIGH;
	int ndata = 0;
	int data[10]; /* 8 data bits, one parity bit, one stop bit */
	int a_bit;

	/* check if communications if inhibited */
	while (input(keyboard_device_clock) == LOW);

	/* wait a while */
	delay_us(100);

	/* set data bits and compute parity bit */
	for (i = 0; i < 8; i++) {
		a_bit = bit_test(packet, i);
		parity ^= a_bit;
		data[ndata++] = a_bit;
	}

	/* set partity bit */
	data[ndata++] = parity;

	/* set stop bit */
	data[ndata++] = HIGH;
	output_high(MARK_PIN);
	output_high(keyboard_device_led);

	/* generate request-to-send */
	output_low(keyboard_device_clock);
	delay_us(65);
	output_low(keyboard_device_data);
	delay_us(65);
	output_high(keyboard_device_clock);

	output_low(MARK_PIN);
	output_high(MARK_PIN);

	/* send the eight data, parity and stop bits */
	for (i = 0; i < ndata; i++) {  
		while (input(keyboard_device_clock) != LOW);	/* wait for clock pulse */

		delay_us(10);
		output_bit(keyboard_device_data, data[i]);

		while (input(keyboard_device_clock) != HIGH);	/* wait for clock pulse */
	}

	/* wait for acknowledgement from device */
	while (input(keyboard_device_data) != LOW);
	while (input(keyboard_device_clock) != LOW);

	while (input(keyboard_device_clock) != HIGH);
	while (input(keyboard_device_data) != HIGH);

	output_low(MARK_PIN);
	output_low(keyboard_device_led);
}

/* ------------------------------------------------------------------------- */

/*
 * This routine receives a packet from the device
 *
 *		-- this routine acts as a host
 */

unsigned int receive_from_device_keyboard(void)
{
	int start;
	int i;
	int data[10]; 	/* 8 data bits, one parity bit, one stop bit */
	int calc_parity = HIGH;
	int parity;
	int stop;
	unsigned int packet = 0x00;

	/* wait for start bit */
	while (input(keyboard_device_clock) != LOW);

	output_high(MARK_PIN);
	output_high(keyboard_device_led);

	start = input(keyboard_device_data);
	assert (start == LOW);

	while (input(keyboard_device_clock) != HIGH);
	
	/* read the eight data bits */
	for (i = 0; i < 10; i++) {
		while (input(keyboard_device_clock) != LOW);

		if (input(keyboard_device_data) == HIGH) {		/* compiler bug: input() */
			data[i] = HIGH;	  /* does not like to assign to an array element */
		} else {
			data[i] = LOW;
		}

		while (input(keyboard_device_clock) != HIGH);
	}

	output_low(MARK_PIN);
	output_low(keyboard_device_led);

	parity = data[8];
	stop = data[9];

	/* compute parity and set bits in packet */
	for (i = 0; i < 8; i++) {
		calc_parity ^= data[i];
		if (data[i] == HIGH) {
			bit_set(packet, i);
		}
	}
	assert (parity == calc_parity);
	assert (stop == HIGH);
	
#if DEBUG == 1
	/* dump the bits */
	printf("receive_host()\r\n");
	printf("start: %d\r\n", start);
	for (i = 0; i < 8; i++) {
		printf("bit[%d]: %d\r\n", i, bit[i]);
	}
	printf("parity: %d\r\n", parity);
	printf("stop: %d\r\n", stop);

	printf("packet: %02X\r\n", packet);
#endif

	return packet;
}

/* ------------------------------------------------------------------------- */

/* ############### generated on Thu Apr 29 09:15:01 EDT 2010 ################ */

